#!/usr/bin/python3

######
## This script takes a json file of paired issues and translates the attachment and comment ids to the migrated system.
## THIS SCRIPT IS NOT INTENDED TO BE USED FOR CLOUD MIGRATION: https://help.k15t.com/backbone-issue-sync/5.8/cloud/server-to-server-with-backbone-cloud-to-server-wit
######

import requests, json, sys, getopt, base64

### parsing command line arguments
def getOptions(argv):
	def printUsage():
		print(f'{sys.argv[0]} -f <mappingFile> --bu1 <urlToFirstJira> --u1 <userFirstJira> --p1 <passwordFirstJira> --bu2 <urlToSecondJira> --u2 <userSecondJira> --p2 <passwordSecondJira>')
		print('If the lookup needs to be performed on the secondConnector, please specify the "-r" option for reversing the connectors.')
		print('If the credential used is personal access token, please specify the "-t" option.')

	try:
		opts, args = getopt.getopt(argv, 'hrtf:', ['file=', 'bu1=', 'u1=', 'p1=', 'bu2=', 'u2=', 'p2=', 'baseUrl1=', 'user1=', 'password1=', 'baseUrl2=', 'user2=', 'password2='])
	except getopt.GetoptError as e:
		print(str(e))
		printUsage()
		sys.exit(2)

	reverseConnectors = False
	useToken = False

	for opt, arg in opts:
		if opt == '-h':
			printUsage()
			sys.exit()
		elif opt == '-r':
			reverseConnectors = True
		elif opt == '-t':
			useToken = True
		elif opt in ('-f', '--file'):
			mappingFile = arg
		elif opt in ('--bu1', '--baseUrl1'):
			baseUrl1 = arg
		elif opt in ('--u1', '--user1'):
			user1 = arg
		elif opt in ('--p1', '--password1'):
			password1 = arg
		elif opt in ('--bu2', '--baseUrl2'):
			baseUrl2 = arg
		elif opt in ('--u2', '--user2'):
			user2 = arg
		elif opt in ('--p2', '--password2'):
			password2 = arg
	if 'mappingFile' not in locals():
		print('mapping file is missing')
		printUsage()
		sys.exit(2)
	if 'baseUrl1' not in locals() or 'user1' not in locals() or 'password1' not in locals():
		print('connection information for first Jira is missing')
		printUsage()
		sys.exit(2)
	if 'baseUrl2' not in locals() or 'user2' not in locals() or 'password2' not in locals():
		print('connection information for second Jira is missing')
		printUsage()
		sys.exit(2)

	return mappingFile, reverseConnectors, useToken, baseUrl1, user1, password1, baseUrl2, user2, password2

### checking if the data for this connector contains deleted/not visible issues
def processConnector(connectorMappingData, baseUrl, user, password, baseUrl2, user2, password2):
	mappingsPerIssue = {}
	# iterate over firstConnector
	for issueData in connectorMappingData:
		try:
			localIssueKey = issueData.get('localIssueKey')
			print(f'processing {localIssueKey}')
			if localIssueKey:
				firstStatusCode, firstAttachmentsMap, firstCommentsMap = getAttachmentsAndComments(localIssueKey, getIdKey, getIdKey, baseUrl, user, password)
				if firstStatusCode == 200:
					secondStatusCode, secondAttachmentsMap, secondCommentsMap = getAttachmentsAndComments(localIssueKey, getFilenameCreatedKey, getBodyCreatedKey, baseUrl2, user2, password2)
					if secondStatusCode == 200:
						attachmentIdMapping = getAttachmentIdMapping(firstAttachmentsMap, secondAttachmentsMap, issueData)
						commentIdMapping = getCommentIdMapping(firstCommentsMap, secondCommentsMap, issueData)
						mappingsPerIssue[localIssueKey] = {'attachmentIdMapping': attachmentIdMapping, 'commentIdMapping': commentIdMapping}
		except Exception as e:
			print(f'Error processing {localIssueKey}: {e}')
	return mappingsPerIssue

def getAttachmentsAndComments(localIssueKey, attachmentKeyFunction, commentKeyFunction, baseUrl, user, password):
	if useToken:
		print('using token')
		authValue = f'Bearer {password}'
	else:
		base64Credentials = base64.b64encode(f'{user}:{password}'.encode('utf-8')).decode('utf-8')
		authValue = f'Basic {base64Credentials}'
	headers = {
		'Authorization': f'{authValue}'
	}
	issueResponse = requests.get(f'{baseUrl}/rest/api/2/issue/{localIssueKey}?fields=attachment,comment', headers=headers, verify=False)
	# print(f'response {issueResponse.text}')
	status_code = issueResponse.status_code

	if status_code == 200:
		issue = issueResponse.json()
		fields = issue.get('fields')
		if fields:
			attachmentsMap = {}
			for attachment in fields.get('attachment'):
				#print(f'attachment with id {attachment['id']} and name {attachment['filename']}')
				attachmentsMap[attachmentKeyFunction(attachment)] = attachment
			commentsMap = {}
			comment = fields.get('comment')
			if comment:
				for comment in comment.get('comments'):
					#print(f'comment with id {comment['id']} and name {comment['body']}')
					comment['body'] = comment['body'].replace('\r\n', '\n')
					commentsMap[commentKeyFunction(comment)] = comment				
		return status_code, attachmentsMap, commentsMap
	else:
		print(f'issue {localIssueKey} not found in {baseUrl}')
		return status_code, None, None

def getIdKey(attachmentOrComent):
	return attachmentOrComent['id']

def getFilenameCreatedKey(attachment):
	return f'{attachment["filename"]}-{attachment["created"]}'

def getBodyCreatedKey(comment):
	return f'{comment["body"]}-{comment["created"]}'

def getAttachmentIdMapping(firstAttachmentsMap, secondAttachmentsMap, issueData):
	attachmentIdMapping = {}
	# Iterate over attachments in firstAttachmentsMap
	for firstAttachmentId, firstAttachment in firstAttachmentsMap.items():
		filename = getFilenameCreatedKey(firstAttachment)
		# Check if filename exists in secondAttachmentsMap
		if filename in secondAttachmentsMap:
			secondAttachment = secondAttachmentsMap[filename]
			secondAttachmentId = secondAttachment['id']
			attachmentIdMapping[firstAttachmentId] = secondAttachmentId
		else:
			print(f'No mapping found for attachment with filename: {filename}')
			if 'attachments' in issueData:
				for attachment in issueData.get('attachments'):
					if 'localId' in attachment:
						localId = attachment.get('localId')
						if localId in attachmentIdMapping:
							print(f'Mapping attachment: {localId} --> {attachmentIdMapping[localId]}')
							attachment['localId'] = attachmentIdMapping[localId]
						else:
							print(f'No attachment found with localId: {localId}')
	return attachmentIdMapping

def getCommentIdMapping(firstCommentsMap, secondCommentsMap, issueData):
	commentIdMapping = {}
	# Iterate over comments in firstCommentsMap
	for firstCommentId, firstComment in firstCommentsMap.items():
		body = getBodyCreatedKey(firstComment)
		# Check if body exists in secondCommentsMap
		if body in secondCommentsMap:
			secondComment = secondCommentsMap[body]
			secondCommentId = secondComment['id']
			commentIdMapping[firstCommentId] = secondCommentId
		else:
			print(f'No mapping found for comment with body: {body}')
			if 'comments' in issueData:
				for comment in issueData.get('comments'):
					if 'localId' in comment:
						localId = comment.get('localId')
						if localId in commentIdMapping:
							print(f'Mapping comment: {localId} --> {commentIdMapping[localId]}')
							comment['localId'] = commentIdMapping[localId]
						else:
							print(f'No comment found with localId: {localId}')
	return commentIdMapping

def replaceIds(keyToReplace, issueKeyType, connectorMappingData, mappingsPerIssue):
	for issueData in connectorMappingData:
		issueKey = issueData.get(issueKeyType)
		print(f'processing {issueKey}')

		if issueKey in mappingsPerIssue:
			mappingPerIssue = mappingsPerIssue[issueKey]
			attachmentIdMapping = mappingPerIssue.get('attachmentIdMapping')
			commentIdMapping = mappingPerIssue.get('commentIdMapping')

			if 'attachments' in issueData:
				for attachment in issueData.get('attachments'):
					if keyToReplace in attachment:
						remoteId = attachment.get(keyToReplace)
						if remoteId in attachmentIdMapping:
							print(f'Mapping attachment: {remoteId} --> {attachmentIdMapping[remoteId]}')
							attachment[keyToReplace] = attachmentIdMapping[remoteId]
						else:
							print(f'No attachment found with id: {remoteId}')

			if 'comments' in issueData:
				for comment in issueData.get('comments'):
					if keyToReplace in comment:
						remoteId = comment.get(keyToReplace)
						if remoteId in commentIdMapping:
							print(f'Mapping comment: {remoteId} --> {commentIdMapping[remoteId]}')
							comment[keyToReplace] = commentIdMapping[remoteId]
						else:
							print(f'No comment found with id: {remoteId}')

def saveJson(jsonData, filename):
	print(f'saving json to {filename}')

	with open(filename, 'w') as f:
		json.dump(jsonData, f, indent=4)

### start of script
mappingFile, reverseConnectors, useToken, baseUrl1, user1, password1, baseUrl2, user2, password2 = getOptions(sys.argv[1:])

with open(mappingFile) as file:
	mappingData = json.load(file)
	
if reverseConnectors:
	connector1 = 'secondConnector'
	connector2 = 'firstConnector'
else:
	connector1 = 'firstConnector'
	connector2 = 'secondConnector'
	
print(f'### Processing sync infos for {connector1}')
mappingsPerIssue = processConnector(mappingData[connector1], baseUrl1, user1, password1, baseUrl2, user2, password2)

if connector2 in mappingData and mappingsPerIssue:
	print(f'### Replacing remote ids for {connector2}')
	replaceIds('remoteId', 'remoteIssueKey', mappingData[connector2], mappingsPerIssue)
	replaceIds('localId', 'localIssueKey', mappingData[connector1], mappingsPerIssue)
	saveJson(mappingData, mappingFile + '-converted.json')