#!/usr/bin/python3

######
## This script removes duplicated attachments and their associated comments from issues
######

import requests, json, sys, getopt
from datetime import datetime


### Glue together two JQL strings
def appendToJql(jql, append):
	if jql == "":
		jql = append
	else:
		jql += f" AND {append}"

	return jql


### Parse command line arguments
def getOptions(argv, scriptName):
	def printApiTokenHelp():
		print("Find more information about Jira API tokens here: https://support.atlassian.com/atlassian-account/docs/manage-api-tokens-for-your-atlassian-account/#Create-an-API-token")

	def printUsage():
		print(f"python3 {scriptName} --url=<urlToJira> --user=<jiraUser> --apiToken=<jiraApiToken> --project=<jiraProject> [--jql=<jqlFilter>] [--dry-run]")
		print()
		printApiTokenHelp()

	try:
		opts, args = getopt.getopt(argv,"h",["help", "url=", "user=", "apiToken=", "jql=", "project=", "dry-run"])
	except getopt.GetoptError as e:
		print(str(e), file=sys.stderr)
		printUsage()
		sys.exit(2)

	dryRun = False
	filterJql = ""

	for opt, arg in opts:
		if opt in ("-h", "--help"):
			printUsage()
			sys.exit()
		elif opt == "--url":
			jiraUrl = arg
		elif opt == "--user":
			jiraUser = arg
		elif opt == "--apiToken":
			jiraPassword = arg
		elif opt == "--jql":
			filterJql = appendToJql(filterJql, arg)
		elif opt == "--project":
			filterJql = appendToJql(filterJql, f"project = '{arg}'")
			projectKey = arg
		elif opt == "--dry-run":
			print("*** Dry run, will not delete anything ***")
			print()
			dryRun = True

	if "jiraUrl" not in locals():
		print("No Jira URL provided: Missing --url option", file=sys.stderr)
		sys.exit(2)
	if "jiraUser" not in locals():
		print("No Jira user provided: Missing --user option", file=sys.stderr)
		sys.exit(2)
	if "jiraPassword" not in locals():
		print("No Jira API token provided: Missing --apiToken option", file=sys.stderr)
		printApiTokenHelp()
		sys.exit(2)
	if "projectKey" not in locals():
		print("No project provided: Missing --project option", file=sys.stderr)
		sys.exit(2)

	return jiraUrl, jiraUser, jiraPassword, filterJql, dryRun, projectKey


### Fetch issues based on a JQL statement
def fetchIssues(jiraUrl, jiraUser, jiraPassword, jql, offset):
	limit = 100

	print("Fetching issues")
	print()

	response = requests.get(f"{jiraUrl}/rest/api/2/search", params={"fields": ["attachment", "comment"], "startAt": offset, "maxResults": limit, "jql": jql}, auth=(jiraUser, jiraPassword))

	if response.status_code != 200:
		print(f"Unable to search for issues with JQL \"{jql}\" - {getStatusCodeError(response.status_code)}", file=sys.stderr)
		exit(3)

	parsed = response.json()
	return parsed["issues"], offset + limit


### Checks, if a project is a JSM project
def isJsmProject(projectKey):
	response = requests.get(f"{jiraUrl}/rest/api/2/project/{projectKey}", auth=(jiraUser, jiraPassword))

	if response.status_code != 200:
		print(f"Unable to find project \"{projectKey}\": {getStatusCodeError(response.status_code)}")
		exit(3)

	return response.json()["projectTypeKey"] == "service_desk"


### Returns a list of duplicated attachments
def findDuplicateAttachments(issue):
	removeIds = []
	removeAttachments = []

	for attachment in issue["fields"]["attachment"]:
		if removeIds.count(attachment["id"]) > 0:
			continue

		for potentialDuplicate in issue["fields"]["attachment"]:
			if attachment["id"] == potentialDuplicate["id"] or removeIds.count(potentialDuplicate["id"]) > 0 or potentialDuplicate["author"]["displayName"] != "Backbone Issue Sync":
				continue

			if attachment["filename"] == potentialDuplicate["filename"] and attachment["size"] == potentialDuplicate["size"]:
				removeIds.append(potentialDuplicate["id"])
				removeAttachments.append(potentialDuplicate)

	return removeAttachments


### Deletes an attachment
def deleteAttachment(attachment, issueKey):
	response = requests.delete(f"{jiraUrl}/rest/api/2/attachment/{attachment['id']}", auth=(jiraUser, jiraPassword))

	if response.status_code > 299:
		print(f"   Unable to delete attachment {attachment['id']} of issue {issueKey} - {getStatusCodeError(response.status_code)}", file=sys.stderr)


### Find comments, referencing a duplicated attachment
def findDuplicateComments(issue, removedAttachments):
	removeIds = []
	removeCommentsPerAttachment = {}
	removeComments = []

	for removedAttachment in removedAttachments:
		for comment in issue["fields"]["comment"]["comments"]:
			if removeIds.count(comment["id"]) > 0 or comment["author"]["displayName"] != "Backbone Issue Sync":
				continue

			if f"[^{removedAttachment['filename']}]" == comment["body"]:
				if removedAttachment["filename"] not in removeCommentsPerAttachment:
					removeCommentsPerAttachment[removedAttachment["filename"]] = []

				removeIds.append(comment["id"])
				removeCommentsPerAttachment[removedAttachment["filename"]].append(comment)
				break

	# Check if there are fewer comments than attachments. If so, we cannot assume the comments are because of the attachment duplication
	for filename in removeCommentsPerAttachment:
		attachments = [attachment for attachment in removedAttachments if attachment["filename"] == filename]

		if len(removeCommentsPerAttachment[filename]) < len(attachments):
			print(f"   Found fewer associated comments for \"{filename}\" created by Backbone than removed attachments with filename \"{filename}\". Therefore, will not remove any comment with content \"[^{filename}]\" in issue {issue['key']}")
		else:
			removeComments.extend(removeCommentsPerAttachment[filename])

	return removeComments


### Deletes a comment
def deleteComment(comment, issueKey):
	response = requests.delete(f"{jiraUrl}/rest/api/2/issue/{issueKey}/comment/{comment['id']}", auth=(jiraUser, jiraPassword))

	if response.status_code > 299:
		print(f"   Unable to delete comment {comment['id']} of issue {issueKey}: {getStatusCodeError(response.status_code)}", file=sys.stderr)


### Returns readable messages from common HTTP errors
def getStatusCodeError(statusCode):
	if statusCode == 403:
		return f"You do not have permissions to perform this operation - HTTP {statusCode}"
	elif statusCode == 404:
		return f"Object not found - HTTP {statusCode}"
	else:
		return f"Unexpected HTTP error {statusCode}"


### Sorts an array of objects by "created" key
def sortByCreated(objects):
	def getMs(object):
		return int(datetime.strptime(object["created"], "%Y-%m-%dT%H:%M:%S.%f%z").timestamp() * 1000)

	sorted(objects, key=getMs)


### Script entry point
jiraUrl, jiraUser, jiraPassword, filterJql, dryRun, projectKey = getOptions(sys.argv[1:], sys.argv[0])
jsmProject = isJsmProject(projectKey)

if jsmProject:
	print("JSM project detected")
else:
	print("SW project detected")

print(f"Using \"{filterJql}\" to search for issues")
print()

offset = 0
issues, offset = fetchIssues(jiraUrl, jiraUser, jiraPassword, filterJql, offset)

while len(issues) > 0:
	for issue in issues:
		print(issue["key"])
		duplicateAttachments = findDuplicateAttachments(issue)
		sortByCreated(duplicateAttachments)

		if len(duplicateAttachments) == 0:
			continue

		for attachment in duplicateAttachments:
			print(f"   Attachment {attachment['id']}: \"{attachment['filename']}\", created by {attachment['author']['displayName']} on {attachment['created']}")

		duplicateComments = []

		if jsmProject:
			duplicateComments = findDuplicateComments(issue, duplicateAttachments)
			sortByCreated(duplicateComments)

		for comment in duplicateComments:
			print(f"   Comment {comment['id']}: \"{comment['body']}\", created by {comment['author']['displayName']} on {comment['created']}")

		if (dryRun == False):
			for duplicateAttachment in duplicateAttachments:
				deleteAttachment(duplicateAttachment, issue["key"])

			for duplicateComment in duplicateComments:
				deleteComment(duplicateComment, issue["key"])

		print()

	issues, offset = fetchIssues(jiraUrl, jiraUser, jiraPassword, filterJql, offset)

print("Done")